import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.bluetooth.RemoteDevice;

import lejos.nxt.Button;
import lejos.nxt.ButtonListener;
import lejos.nxt.Motor;
import lejos.nxt.Sound;
import lejos.nxt.comm.BTConnection;
import lejos.nxt.comm.Bluetooth;
import lejos.nxt.comm.NXTConnection;
import lejos.nxt.comm.USB;

/**
 * This application is the main control app on the middle NXT, it manages the X, Y and 
 * base rotation motors, along with receiving models from the computer and sending 
 * commands to the other two NXTs.
 *
 * @author Will Gorman (willgorman@hotmail.com)
 **/
public class MakerLegoBotXY {

	static final int SPC = 114;
	static final int X_OFFSET = -212; // MOTOR A OFFSET
	static final int Y_OFFSET =  110; // MOTOR B OFFSET
	static final int brickTypeLocation[] = {3, 6, 10, 0, 16};
	static final int WIGGLE = 40;
       
	// NXT's from Lego:
	static final String NXT_NAMES[] = {"NXT", "Jette"};
	// NXT's at Home: 
        // static final String NXT_NAMES[] = {"NXT2", "NXT"};
	
	static final int _2x1 = 0;
	static final int _2x2 = 1;
	static final int _2x3 = 2;
	static final int _2x4 = 3;
	static final int _2x8 = 4;
	
	static BTConnection btc[] = new BTConnection[2];
	static DataInputStream dis[] = new DataInputStream[2];
	static DataOutputStream dos[] = new DataOutputStream[2];
	static int currHeight = 0;
	static int x_align = 0;
	static int y_align = 0;
	static int redo = 0;

	public static void establishDataStream(int index, String device) throws Exception {
		Util.drawString("Estab DS " + device);
		RemoteDevice rd = Bluetooth.getKnownDevice(device);
		if (rd == null) {
			Util.drawString("No such device");
		    Button.waitForPress();
		    System.exit(1);
		}

		btc[index] = Bluetooth.connect(rd);
    		if (btc[index] == null) {
			Util.drawString("Connect fail");
			Button.waitForPress();
			System.exit(1);
		}
		
		dis[index] = btc[index].openDataInputStream();
		dos[index] = btc[index].openDataOutputStream();
		Util.drawString("Done Estab DS " + device);
	}

	public static void establishDataStreams() throws Exception {
		establishDataStream(0, NXT_NAMES[0]);
		establishDataStream(1, NXT_NAMES[1]);
	}

	public static void execRemoteCommand(int device, String commandName, int commandCode) throws Exception {
		Util.drawString(commandName);
		// SEND COMMAND TO GET A BRICK
		dos[device].writeInt(commandCode);
		dos[device].flush();
		Util.drawString("Sent " + commandName + " Waiting");
		// WAIT TILL BRICK IS COMPLETE
		int r = dis[device].readInt();
		Util.drawString("Completed " + commandName);
	}

	public static void getABrick() throws Exception {
		execRemoteCommand(0, "getABrick", 0);
	}

	public static void placeABrick() throws Exception {
		execRemoteCommand(0, "placeABrick", 1);
		try {
			handleState();
			if (state == ALIGN) {
				int x = Motor.A.getTachoCount();
				int y = Motor.B.getTachoCount();
				while (state == ALIGN) {
					Util.drawString("ALIGN");
					Thread.sleep(100);
				}
				x_align = Motor.A.getTachoCount() - x + x_align;
				y_align = Motor.B.getTachoCount() - y + y_align;
				Util.drawString("align: " + x_align + ", " + y_align + " ENTER TO CONTINUE");
				state = PAUSE;
				while (state == PAUSE) {
					Thread.sleep(100);
				}
				state = RUN;
			}		
			// wiggle y
			int tacho = Motor.B.getTachoCount();
			Motor.B.rotateTo(tacho + WIGGLE);
			Motor.B.rotateTo(tacho - WIGGLE);
			Motor.B.rotateTo(tacho);
			handleState();
		} finally {
			execRemoteCommand(0, "releaseABrick", 6);
		}
		handleState();
	}

	public static void pressBrick() throws Exception {
		execRemoteCommand(0, "pressBrick", 5);
	}

	public static void backOff() throws Exception {
		execRemoteCommand(1, "backOff", 0);
	}
	
	public static void lockIn() throws Exception {
		execRemoteCommand(1, "lockIn", 1);
	}

	public static void goToZ(int val) throws Exception {
		execRemoteCommand(1, "goToZ " + val, val + 3);
	}

	public static void disconnect(int index) throws Exception {
		Util.drawString("disconnect " + index);
		try {
			dos[index].writeInt(2);
			dos[index].flush();
			dis[index].close();
			dos[index].close();
			btc[index].close();
		} catch (IOException ioe) {
			Util.drawString("Close Exception");
		}
	}

	public static void disconnect() throws Exception {
		disconnect(0);
		disconnect(1);
	}

	public static void rotatePlateCounterClockwise() {
		Motor.C.rotateTo(Motor.C.getTachoCount() + 90);
	}
	public static void rotatePlateClockwise() {
		Motor.C.rotateTo(Motor.C.getTachoCount() - 90);
	}

	public static class ActionInterrupt extends Exception {
	}

	public static void getBrick(int brickType) throws Exception {
		boolean debugGetABrick = false;
		if (Motor.A.getTachoCount() != 0 || Motor.B.getTachoCount() != SPC * brickTypeLocation[brickType]) {
			if (Motor.A.getTachoCount() > -140) {
				Motor.A.rotateTo(-140);
			}
			Motor.B.rotateTo(SPC * brickTypeLocation[brickType]);
			Motor.A.rotateTo(0);
		}
		handleState();
		if (!debugGetABrick) {
			getABrick();
		} else { 
			Button.ENTER.waitForPressAndRelease();
			execRemoteCommand(0, "pickupLegoA", 3);
			Button.ENTER.waitForPressAndRelease();
			execRemoteCommand(0, "pickupLegoB", 4);
			Button.ENTER.waitForPressAndRelease();
			getABrick();
		}
	}

	public static void handleState() throws Exception {
		if (state == RUN && redo > 0) {
			throw new ActionInterrupt();
		}
		if (state == RUN || state == ALIGN) return;
		while (state == PAUSE) {
			Util.drawString("PAUSED");
			Thread.sleep(100);
		}
		while (state == DONE) {
			Util.drawString("DONE, ENTER TO LOWER");
			Thread.sleep(100);
		}
		if (state == CANCEL) {
			throw new Exception("CANCEL");
		}
		if (state == STOP) {
			throw new Exception("STOP");
		}
	}

	public static void getAndPlaceBrick(int type, int x, int y) throws Exception {
		handleState();
		getBrick(type);
		handleState();
		Motor.A.rotateTo(-SPC * x + X_OFFSET + x_align);
		// if true, this is bad, someone is using an x value out of bounds
		if (Motor.A.getTachoCount() > -140) {
			Motor.A.rotateTo(-140);
		}
		handleState();
		Motor.B.rotateTo(SPC * y + Y_OFFSET + y_align);
		handleState();
		placeABrick();
		handleState();
		Motor.A.rotateTo(-SPC * (x - 3) + X_OFFSET + x_align);
		handleState();
		pressBrick();
		if (type == _2x8) {
			handleState();
			Motor.B.rotateTo(SPC * (y - 2) + Y_OFFSET + y_align);
			handleState();
			pressBrick();
			handleState();
			Motor.B.rotateTo(SPC * (y + 2) + Y_OFFSET + y_align);
			handleState();
			pressBrick();
		}
		if (Motor.A.getTachoCount() > -140) {
			Motor.A.rotateTo(-140);
		}
		handleState();
	}

	public static void rotate(int num) throws Exception {
		handleState();
		goToZ(currHeight + 3);
		handleState();
		backOff();
		handleState();
		for (int i = 0; i < num; i++) {
			rotatePlateClockwise();
			handleState();
		}
		lockIn();
		handleState();
		goToZ(currHeight);
		handleState();
	}

	public static void rotateCounterClockwise(int num) throws Exception {
		handleState();
		goToZ(currHeight + 3);
		handleState();
		backOff();
		handleState();
		for (int i = 0; i < num; i++) {
			rotatePlateCounterClockwise();
			handleState();
		}
		lockIn();
		handleState();
		goToZ(currHeight);
		handleState();
	}


	public static void house() throws Exception {
		Util.drawString("Printing House");
		
		getAndPlaceBrick(_2x4, 2, 9);

		rotate(1);

		getAndPlaceBrick(_2x4, 5, 5);
		getAndPlaceBrick(_2x4, 5, 9);

		rotate(3);

		getAndPlaceBrick(_2x4, 12, 9);

		rotate(1);

		getAndPlaceBrick(_2x4, 9, 3);
		getAndPlaceBrick(_2x2, 9, 6);
		getAndPlaceBrick(_2x4, 9, 11);

		// SECOND LEVEL

		goToZ(1);
		currHeight = 1;

		getAndPlaceBrick(_2x4, 5, 3);
		getAndPlaceBrick(_2x4, 5, 7);
		getAndPlaceBrick(_2x4, 5, 11);
		
		rotate(1);
		getAndPlaceBrick(_2x4, 3, 8);

		rotate(3);
		getAndPlaceBrick(_2x4, 9, 5);
		getAndPlaceBrick(_2x2, 9, 10);

		rotate(1);
		getAndPlaceBrick(_2x4, 13, 8);

		
		goToZ(2);
		currHeight = 2;
		rotate(2);
		
		getAndPlaceBrick(_2x4, 2, 9);

		rotate(1);

		getAndPlaceBrick(_2x4, 5, 5);
		getAndPlaceBrick(_2x4, 5, 9);

		rotate(3);

		getAndPlaceBrick(_2x4, 12, 9);

		rotate(1);

		getAndPlaceBrick(_2x4, 9, 3);
		getAndPlaceBrick(_2x8, 9, 9);

	}

	public static void testBricks() throws Exception {
				Util.drawString("Printing TestBricks");

		// Sent Command: B: 2x1, 10, 7
		// Sent Command: B: 2x3, 10, 6
		getAndPlaceBrick(_2x1, 2, 5);
		getAndPlaceBrick(_2x1, 4, 5);
		getAndPlaceBrick(_2x1, 6, 5);
		getAndPlaceBrick(_2x1, 8, 5);
		getAndPlaceBrick(_2x1, 10, 5);
		getAndPlaceBrick(_2x1, 12, 5);
		if (true) return;

		getAndPlaceBrick(_2x1, 1, 11);
		getAndPlaceBrick(_2x3, 1, 7);

		getAndPlaceBrick(_2x1, 4, 11);
		getAndPlaceBrick(_2x3, 4, 7);

		getAndPlaceBrick(_2x1, 7, 11);
		getAndPlaceBrick(_2x3, 7, 7);

		getAndPlaceBrick(_2x1, 10, 11);
		getAndPlaceBrick(_2x3, 10, 7);

		getAndPlaceBrick(_2x1, 13, 11);
		getAndPlaceBrick(_2x3, 13, 7);


		if (true) return;

		// What is the full capable range of this robot?
		getAndPlaceBrick(_2x2, 2, 0);
		getAndPlaceBrick(_2x2, 2, 2);
		getAndPlaceBrick(_2x2, 4, 4);
		// getAndPlaceBrick(_2x2, 6, 6);
		// getAndPlaceBrick(_2x2, 8, 8);
		// getAndPlaceBrick(_2x2, 10, 10);
		// getAndPlaceBrick(_2x2, 12, 12);
		// 
		getAndPlaceBrick(_2x2, 12, 14);
		getAndPlaceBrick(_2x2, 12, 16);
		// getAndPlaceBrick(_2x2, 18, 18);



		// getAndPlaceBrick(_2x4, 0, 0);
		// getAndPlaceBrick(_2x4, 3, 3);
		getAndPlaceBrick(_2x4, 6, 6);
		getAndPlaceBrick(_2x4, 9, 9);
		//getAndPlaceBrick(_2x4, 12, 12);

		// getAndPlaceBrick(_2x4, 3, 0);
		// getAndPlaceBrick(_2x4, 6, 3);
		// getAndPlaceBrick(_2x4, 9, 6);
		// getAndPlaceBrick(_2x4, 12, 9);

		// 		goToZ(1);

		// getAndPlaceBrick(_2x4, 3, 0);
		// getAndPlaceBrick(_2x4, 6, 3);
		// getAndPlaceBrick(_2x4, 9, 6);
		// getAndPlaceBrick(_2x4, 12, 9);


		// goToZ(2);

		// getAndPlaceBrick(_2x4, 3, 0);
		// getAndPlaceBrick(_2x4, 6, 3);
		// getAndPlaceBrick(_2x4, 9, 6);
		// getAndPlaceBrick(_2x4, 12, 9);


		/*
		getAndPlaceBrick(_2x1, 3, 0);
		getAndPlaceBrick(_2x2, 3, 2);
		getAndPlaceBrick(_2x3, 3, 6);
		getAndPlaceBrick(_2x4, 3, 10);
		getAndPlaceBrick(_2x8, 6, 3);
		*/
	}

	public static void pyramid() throws Exception {
		Util.drawString("Printing Pyramid");
		// Move to grab a 2x2 block
		getAndPlaceBrick(_2x2, 4, 4);
		getAndPlaceBrick(_2x2, 4, 6);
		getAndPlaceBrick(_2x2, 4, 8);
		getAndPlaceBrick(_2x2, 6, 4);
		getAndPlaceBrick(_2x2, 6, 6);
		getAndPlaceBrick(_2x2, 6, 8);
		getAndPlaceBrick(_2x2, 8, 4);
		getAndPlaceBrick(_2x2, 8, 6);
		getAndPlaceBrick(_2x2, 8, 8);
		goToZ(1);
		getAndPlaceBrick(_2x2, 5, 5);
		getAndPlaceBrick(_2x2, 5, 7);
		getAndPlaceBrick(_2x2, 7, 5);
		getAndPlaceBrick(_2x2, 7, 7);
		goToZ(2);
		getAndPlaceBrick(_2x2, 6, 6);
	}

	public static void hollowPyramid() throws Exception {
		Util.drawString("Printing Hollow Pyramid");
		// Move to grab a 2x2 block
		
		getAndPlaceBrick(_2x2, 4, 4);
		getAndPlaceBrick(_2x2, 4, 6);
		getAndPlaceBrick(_2x2, 4, 8);
		getAndPlaceBrick(_2x2, 6, 4);
		getAndPlaceBrick(_2x2, 6, 8);
		getAndPlaceBrick(_2x2, 8, 4);
		getAndPlaceBrick(_2x2, 8, 6);
		getAndPlaceBrick(_2x2, 8, 8);
		goToZ(1);
		getAndPlaceBrick(_2x2, 5, 5);
		getAndPlaceBrick(_2x2, 5, 7);
		getAndPlaceBrick(_2x2, 7, 5);
		getAndPlaceBrick(_2x2, 7, 7);
		goToZ(2);
		getAndPlaceBrick(_2x2, 6, 6);
	}

	public static interface Action {
		public void doAction() throws Exception;
	}

	public static class Rotate implements Action {
		boolean clockwise = true;
		public Rotate(boolean clockwise) {
			this.clockwise = clockwise;
		}
		public void doAction() throws Exception {
			if (clockwise) {
				rotate(1);
			} else {
				rotateCounterClockwise(1);
			}
		}	
	}

	public static class PlaceBrick implements Action {
		int brick;
		int x;
		int y;

		public PlaceBrick(int brick, int x, int y) {
			this.brick = brick;
			this.x = x;
			this.y = y;
		}

		public void doAction() throws Exception {
			getAndPlaceBrick(brick, x, y);
		}
	}
	public static class GoToZ implements Action {
		int z;
		public GoToZ(int z) {
			this.z = z;
		}
		public void doAction() throws Exception {
			goToZ(z);
			currHeight = z;
		}
	}

	public static void fromPC() throws Exception {
		// download execution plan from PC
		Sound.beep();
		Util.drawString("Waiting for PC");
		NXTConnection connection = USB.waitForConnection();
		DataInputStream dis = connection.openDataInputStream();
		// COMMANDS
		// Z (Z Location)
		// B (Brick) 
		// R (Rotate)
		// D (Done)
		List<Action> actions = new ArrayList<Action>();
		char c = dis.readChar();
		while (c != 'D') {
			Util.drawString("Adding " + c + " T:" + actions.size());
			if (c == 'Z') {
				// Z Value, read location
				actions.add(new GoToZ(dis.readInt()));
				Util.drawString("Added Z");
			} else if (c == 'B') {
				actions.add(new PlaceBrick(dis.readInt(), dis.readInt(), dis.readInt()));
				Util.drawString("Added B");
			} else if (c == 'R') {
				actions.add(new Rotate(dis.readBoolean()));
				Util.drawString("Added R");
			} else {
				Util.drawString("Invalid: " + c + actions.size());
				c = 'D';
			}
			if (c != 'D') {				
				c = dis.readChar();
			}
		}
		dis.close();
		connection.close();
		Util.drawString("Executing " + actions.size() + " Actions");
		// build structure
		for (int i = 0; i < actions.size(); i++) {
			try {
				Action action = actions.get(i);
				action.doAction();
				handleState();
			} catch (ActionInterrupt e) {
			}
			if (redo > 0) {
				i -= redo;
				if (i < 0) i = 0;
				redo = 0;
			}
		}
		/*
		for (Action action : actions) {
			action.doAction();
			handleState();
		}
		*/
	}

	public static void startup() throws Exception {
		// Wait for manual alignment completion
		establishDataStreams();
		// Util.drawString("Man Conf, ENTER");
		// Button.ENTER.waitForPressAndRelease();
		Util.resetTachos();
	}

	public static void shutdown() throws Exception {
		state = DONE;
		boolean zMove = true;
		// Go to the top
		if (zMove) {
			goToZ(13);
		}
		if (Motor.A.getTachoCount() > -140) {
			Motor.A.rotateTo(-140);
		}
		Motor.B.rotateTo(0);
		Motor.A.rotateTo(0);

		if (zMove) {
			// wait for enter
			Util.drawString("Build Complete, ENTER");
			handleState();
			// go back down		
			goToZ(0);
		}
		disconnect();
	}

	static int state = -1;
	static final int INIT = -1;
	static final int RUN = 0;
	static final int PAUSE = 1;
	static final int CANCEL = 2;
	static final int STOP = 3;
	static final int DONE = 5;
	static final int LOWER = 6;
	static final int ALIGN = 7;

	static int model = 0;
	static int HOUSE = 4;
	static int PYRAMID = 1;
	static int TESTBRICKS = 2;
	static int HOLLOW_PYRAMID = 3;
	static int FROM_PC = 0;
	static int NOMODEL = 5;
	
	static String models[] = {"From PC", "Pyramid", "Test Bricks", "Hollow Pyramid", "House", "No Model"};

	public static void main (String[] args) throws Exception {

		Button.ENTER.addButtonListener(new ButtonListener() {
			public void buttonPressed(Button b) {
				if (state == INIT) {
					state = RUN;
				} else if (state == RUN) {
					state = PAUSE;
				} else if (state == ALIGN) {
					state = RUN;
				} else if (state == PAUSE) {
					state = RUN;
				} else if (state == DONE) {
					state = LOWER;
				}
			}
			public void buttonReleased(Button b) {}
		});
		Button.ESCAPE.addButtonListener(new ButtonListener() {
			public void buttonPressed(Button b) {
				if (state == INIT) {
					System.exit(0);
				} else if (state == RUN) {
					state = CANCEL;
				} else if (state == CANCEL) {
					state = STOP;
				}
			}
			public void buttonReleased(Button b) {}		
		});
		Button.LEFT.addButtonListener(new ButtonListener() {
			public void buttonPressed(Button b) {
				if (state == INIT) {
					model--;
					if (model == -1) model = models.length - 1;
					Util.drawString("MODEL: " + models[model]);
				} else 	if (state == PAUSE) {
					redo++;
					Util.drawString("REDO: " + redo);
				}
			}
			public void buttonReleased(Button b) {}
		});
		Button.RIGHT.addButtonListener(new ButtonListener() {
			public void buttonPressed(Button b) {
				if (state == INIT) {
					model++;
					if (model == models.length) model = 0;
					Util.drawString("MODEL: " + models[model]);
				} else if (state == PAUSE) {
					state = ALIGN;
				Util.drawString("ALIGN");
				}
			}
			public void buttonReleased(Button b) {}
		});
		Util.drawString("MODEL: From PC (ENTER)");
		while (state == INIT) {
			Thread.sleep(100);
		}
		if (state == RUN) {
			startup();
			try {
				if (model == HOUSE) {
					house();
				} else if (model == PYRAMID) {
					pyramid();
				} else if (model == TESTBRICKS) {
					testBricks();
				} else if (model == HOLLOW_PYRAMID) {
					hollowPyramid();
				} else if (model == FROM_PC) {
					fromPC();
				} else if (model == models.length - 1) {
					// do nothing
				} else {
					Util.drawString("NO MODEL SELECTED");
				}
			} catch (Exception e) {
				Util.drawString("E: " + e.getMessage());
				Thread.sleep(5000);
				if (!e.getMessage().equals("CANCEL")) {
					throw e;
				}
			} finally {
				shutdown();

			}
			return;
		}
	}

}
