import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import lejos.pc.comm.NXTCommFactory;
import lejos.pc.comm.NXTConnector;

/**
 * This app takes a MLCad .ldr model, determines a print plan, and then sends
 * the individual print commands to the MakerLegoBot
 * 
 * @author Will Gorman (willgorman@hotmail.com)
 **/
public class PCApp {
	public static void main(String args[]) throws Exception {
		List<Action> actions = getActions(args[0]);
		NXTConnector connector = new NXTConnector();
		
		boolean connected = connector.connectTo("", "", NXTCommFactory.USB);
		if (connected) {
			System.out.println("Connected, Sending Command");
			// SEND COMMAND
			DataInputStream dataIn = connector.getDataIn();
			DataOutputStream dataOut = connector.getDataOut();
			int c = 0;
			// toggle the command to begin on
			int command = 0;
			// optionally send the printer to a Z coordinate before printing
			// new GoToZ(5).sendCommand(dataOut);
			for (Action a : actions) {
				if (c >= command) {
					a.sendCommand(dataOut);
					System.out.println("Sent Command: " + a);
				} else {
					System.out.println("Skipped Command: " + a);
				}
				c++;
			}
			dataOut.writeChar('D');
			dataOut.flush();
			dataIn.close();
			dataOut.close();
			connector.close();
		} else {
			System.out.println("Unable to Connect");
		}
	}
	
	public static boolean isPointInBounds(int x, int y, int bx, int by, int ex, int ey) {
		return x >= bx && y >= by && x <= ex && y <= ey;
	}

	public static class Brick implements Comparable<Brick> {
		String orig;
		int type = 0;
		int rt = NO_ROTATE;
		int x = 0;
		int y = 0;
		int z = 0;

		public boolean isRotated() {
			return rt != NO_ROTATE;
		}
		
		public int getXP() {
			if (rt == ROTATE_CLOCKWISE) {
				return y + relxloc[type];
			} else if (rt == NO_ROTATE){
				return x + relxloc[type];
			} else if (rt == ROTATE_COUNTER_CLOCKWISE) {
				return (13 - y) + relxloc[type];
			}
			throw new UnsupportedOperationException("INVALID ROTATION STATE:" + rt);
		}

		public int getYP() {
			if (rt == ROTATE_CLOCKWISE) {
				return (13 - x) + relyloc[type];
			} else if (rt == NO_ROTATE) {
				return y + relyloc[type];
			} else if (rt == ROTATE_COUNTER_CLOCKWISE) {
				return x + relyloc[type];
			}
			throw new UnsupportedOperationException("INVALID ROTATION STATE:" + rt);
		}

		public boolean isAvailForPrinting(List<Brick> others, int rt) {
			return brickTypes[type].isAvailForPrinting(this, others, rt);
		}
		
		public boolean inBounds(int bx, int by, int ex, int ey) {
			boolean inb = brickTypes[type].inBounds(this, bx, by, ex, ey);
			System.out.println("BOUND CHECK FOR " + toString() + " : " + bx + "," + by + "," + ex + "," + ey + " = " + inb);
			return inb;
		}

		public int compareTo(Brick b) {
			if (z > b.z) {
				return 1;
			} else if (z < b.z) {
				return -1;
			} else if (y > b.y) {
				return 1;
			} else if (y < b.y) {
				return -1;
			} else if (x > b.x) {
				return 1;
			} else if (x < b.x) {
				return -1;
			} else {
				return 0;
			}
		}
		public boolean equals(Object o) {
			Brick b = (Brick)o;
			return b.x == x && b.y == y && b.z == z && b.type == type;
		}

		public String toString() {
			return brickTypes[type].getName() + ": c=" + x + "," + y + "," + z + " p=" + getYP() + "," + getXP() + "," + z + " rt: " + rotationType[rt] + " orig: " + orig;
		}
	}

	static final int _2x1 = 0;
	static final int _2x2 = 1;
	static final int _2x3 = 2;
	static final int _2x4 = 3;
	static final int _2x8 = 4;

	// relative brick x print location
	static final int relxloc[] = {1,1,2,1,1};
	static final int relyloc[] = {1,1,1,1,1};
	static final int revrelxloc[] = {1,1,1,1,1};
	static final int revrelyloc[] = {1,1,-1,-1,-1};
	static final BrickType brickTypes[] = {new Brick2x1(), new Brick2x2(), new Brick2x3(), new Brick2x4(), new Brick2x8()};
	static final int ROTATE_COUNTER_CLOCKWISE = 0;
	static final int NO_ROTATE = 1;
	static final int ROTATE_CLOCKWISE = 2;
	static final String rotationType[] = {"Rotate Counter Clockwise", "No Rotate", "Rotate Clockwise"}; 
	
	static class Brick2x1 extends AbstractBrickType {
		public Brick2x1() {
			name = "2x1";
			modelName = "3004.DAT";
			index = 0;
			ar = new int[][] {{-2,1,2,6},{-2,-1,-3,2}};
			ib = new int[][] {{0,0}};
		}
	}

	static class Brick2x2 extends AbstractBrickType {
		public Brick2x2() {
			name = "2x2";
			modelName = "3003.DAT";
			index = 1;
			ar = new int[][] {{-2,1,2,6},{-2,-1,-3,2}};
			ib = new int[][] {{0,0}};
		}
	}
	
	static class Brick2x3 extends AbstractBrickType {
		public Brick2x3() {
			name = "2x3";
			modelName = "3002.DAT";
			index = 2;
			ar = new int[][] {{-2,1,2,6},{-2,-1,-3,2}};
			ib = new int[][] {{0,0}, {1, 0}};
		}
	}

	static class Brick2x4 extends AbstractBrickType {
		public Brick2x4() {
			name = "2x4";
			modelName = "3001.DAT";
			index = 3;
			ar = new int[][] {{-2,1,2,6},{-2,-1,-3,2}};
			ib = new int[][] {{0,0},{-1,0},{1,0}};
		}
	}
	
	static class Brick2x8 extends AbstractBrickType {
		public Brick2x8() {
			name = "2x8";
			modelName = "3007.DAT";
			index = 4;
			ar = new int[][] {{-4,1,4,6}, {-5,-1,-6,2}};
			ib = new int[][] {{-3,0},{-2,0},{-1,0},{0,0},{1,0},{2,0},{3,0}};
		}
	}

	static interface BrickType {
		public boolean inBounds(Brick b, int bx, int by, int ex, int ey);
		boolean isAvailForPrinting(Brick b, List<Brick> others, int rt);
		public String getName();
		public int getIndex();
		public String getModelName();
	}
	
	static abstract class AbstractBrickType implements BrickType {
		String name;
		String modelName;
		int index;
		int ar[][];
		int ib[][];

		public String getName() {
			return name;
		}
		public String getModelName() {
			return modelName;
		}
		public int getIndex() {
			return index;
		}
		public boolean isAvailForPrinting(Brick b, List<Brick> others, int rt) {
			int xr = 0;
			int yr = 0;
			int rx = 1;
			int ry = 1;
			if ((b.isRotated() && rt == NO_ROTATE) || (!b.isRotated() && rt != NO_ROTATE)) {
				throw new RuntimeException("Assertion that "+ rt + " and " + b.rt + " are compatible failed");
			}

			if (rt == ROTATE_CLOCKWISE) { // b.isRotated()) {
				xr = 1;
				yr = -1;
				
				rx = -1;
				// ry = -1;
			} else if (rt == ROTATE_COUNTER_CLOCKWISE) {
				xr = 1;
				yr = -1;
				// rx = -1;
				ry = -1;
				
			}
			System.out.println("BOUND B: " + b);
			for (Brick o : others) {
				if (o.equals(b)) continue;
				if (o.inBounds(b.x + ar[0][0 + xr] * rx, b.y + ar[0][1 + yr], b.x + ar[0][2 + xr] * rx, b.y + ar[0][3 + yr] * ry) || 
				    o.inBounds(b.x + ar[1][0 + xr] * rx, b.y + ar[1][1 + yr], b.x + ar[1][2 + xr] * rx, b.y + ar[1][3 + yr] * ry)) {
					System.out.println("IS NOT AVAIL");

					return false;
				}
			}
			System.out.println("IS AVAIL");
			return true;
		}
		public boolean inBounds(Brick b, int bx, int by, int ex, int ey) {
			if (bx > ex) {
				int tx = bx;
				bx = ex;
				ex = tx;
			}
			if (by > ey) {
				int ty = by;
				by = ey;
				ey = ty;
			}
			int xi = 0;
			int yi = 1;
			if (b.rt != NO_ROTATE) {
				xi = 1;
				yi = 0;
			}
			boolean inbounds = false;
			for (int i = 0; i < ib.length; i++) {
				inbounds |= isPointInBounds(b.x + ib[i][xi], b.y + ib[i][yi], bx, by, ex, ey);
			}
			return inbounds;
		}
	}
	
	public static int getBrickType(String val) {
		for (int i = 0; i < brickTypes.length; i++) {
			if (val.equals(brickTypes[i].getModelName())) {
				return i;
			}
		} 
		return -1;
	}

	public static interface Action {
		public void sendCommand(DataOutputStream dataOut) throws Exception;
	}

	public static class Rotate implements Action {
		boolean clockwise;
		public Rotate(boolean clockwise) {
			this.clockwise = clockwise;
		}
		public void sendCommand(DataOutputStream dataOut) throws Exception {
			dataOut.writeChar('R');
			dataOut.writeBoolean(clockwise);
			dataOut.flush();
		}
		public String toString() { return "R: " + clockwise; }		
	}
	public static class PlaceBrick implements Action {
		int brick;
		int x;
		int y;

		public PlaceBrick(int brick, int x, int y) {
			this.brick = brick;
			this.x = x;
			this.y = y;
		}
		public void sendCommand(DataOutputStream dataOut) throws Exception {
			dataOut.writeChar('B');
			dataOut.writeInt(brick);
			dataOut.writeInt(x);
			dataOut.writeInt(y);
			dataOut.flush();
		}
		public String toString() { return "B: " + brickTypes[brick].getName() +", " + x +", " + y; }		
	}
	public static class GoToZ implements Action {
		int z;
		public GoToZ(int z) {
			this.z = z;
		}
		public void sendCommand(DataOutputStream dataOut) throws Exception {
			dataOut.writeChar('Z');
			dataOut.writeInt(z);
		}
		public String toString() { return "Z: " + z; }
	}

	public static List<Action> getActions(String filename) throws Exception {
		// Load Bricks from file
		BufferedReader br = new BufferedReader(new FileReader(filename));
		String line = br.readLine();
		List<Brick> bricks = new ArrayList<Brick>();
		int lc = 1;
		int minz = Integer.MAX_VALUE;
		int maxz = Integer.MIN_VALUE;
		int miny = Integer.MAX_VALUE;
		int maxy = Integer.MIN_VALUE;
		int minx = Integer.MAX_VALUE;
		int maxx = Integer.MIN_VALUE;
		while (line != null) {
			
			String vals[] = line.split(" ");
			if (vals[0].equals("1")) {
				// we've got a part to deal with
				Brick brick = new Brick();
				brick.orig = line;
				brick.type = getBrickType(vals[14]);
				if (brick.type == -1) {
					System.out.println("Invalid Brick Type (line " + lc+ "):" + vals[14]);
					return null;
				}
				brick.x = Integer.parseInt(vals[2]) * -1;
				brick.y = Integer.parseInt(vals[4]) * -1;
				brick.z = Integer.parseInt(vals[3]) * -1;
				if (!vals[7].equals("0")) {
					brick.rt = ROTATE_CLOCKWISE;
				}
				// 2x2's don't need to rotate
				if (brick.type == _2x2) brick.rt = NO_ROTATE;
				if (brick.type == _2x1) {
					if (!brick.isRotated()) {
						brick.rt = ROTATE_CLOCKWISE;
					} else {
						brick.rt = NO_ROTATE;
					}
				}
								
				if (brick.z % 24 != 0) {
					System.out.println("Invalid Brick Z Location (line " + lc+ "): " + brick.z + " " + brick.z % 24);
					return null;
				}
				brick.z = brick.z / 24;
				
				brickx:
				if (brick.x % 20 != 0) {
					// we could be dealing with a 2x3 or 2x1?
					if ((brick.isRotated() || true) && (brick.type == _2x3 || brick.type == _2x1) && brick.x % 10 == 0) {
						brick.x = (brick.x - 10) / 20;
						break brickx;
					}
					System.out.println("Invalid Brick X Location (line " + lc+ "): " + brick.x + " " + brick.x % 20 + "\n" + line);
					return null;
				} else {
					brick.x = brick.x / 20;
				}
				
				if (brick.y % 20 != 0) {
					System.out.println("Invalid Brick X Location (line " + lc+ "): " + brick.y + " " + brick.y % 20);
					return null;
				}
				brick.y = brick.y / 20;

				if (brick.x < minx) minx = brick.x;
				if (brick.x > maxx) maxx = brick.x;
				if (brick.y < miny) miny = brick.y;
				if (brick.y > maxy) maxy = brick.y;
				if (brick.z < minz) minz = brick.z;
				if (brick.z > maxz) maxz = brick.z;

				bricks.add(brick);
			}
			lc++;
			line = br.readLine();
		}
		br.close();

		int width = maxx - minx + 1;
		int depth = maxy - miny + 1;
		
		int xoffset = (23 - width) / 2 - 5;
		int yoffset = (23 - depth) / 2 - 5;
		
		System.out.println("WIDTH = " + width + ", DEPTH = " + depth);
		// normalize numbers
		for (Brick b : bricks) {
			b.x = b.x - minx + xoffset;
			b.y = b.y - miny + yoffset;
			b.z -= minz;
		}

		List<List<Brick>> bricklist = getZLists(bricks, maxz-minz);

		List<Action> actions = new ArrayList<Action>();

		// for every in play brick in the current z
		// see if there is any other brick intersecting it's print bounds (see chart)
                // if not, remove from play in stack for printing (reverse order)
		// boolean r = false;
		int rt = NO_ROTATE; // no rotation, rotate left, or rotate right
		int z = 0;
		// rotation state
		int state[][] = {{ROTATE_COUNTER_CLOCKWISE, NO_ROTATE, ROTATE_CLOCKWISE},
				         {NO_ROTATE, ROTATE_CLOCKWISE, ROTATE_COUNTER_CLOCKWISE},
				         {ROTATE_CLOCKWISE, NO_ROTATE, ROTATE_COUNTER_CLOCKWISE}};
		for (List<Brick> bks : bricklist) {
			int crt = rt;
			List<Brick> reverseOrderedBricks = new ArrayList<Brick>();
			while (bks.size() > 0) {
				int c = bks.size();
				// first iteration, look for bricks in current rotation state
				stateLoop:
				for (int i = 0; i < 3; i++) {
					for (Brick b : bks) {
						// if the brick is compatible with the rotation state, see if it is available
						if ( (state[rt][i] == NO_ROTATE && !b.isRotated()) || 
						     (state[rt][i] != NO_ROTATE && b.isRotated())) 
						{
							if (b.isAvailForPrinting(bks, state[rt][i])) { 
								reverseOrderedBricks.add(b);
								// update the rotation state
								rt = b.rt = state[rt][i];
								bks.remove(b);
								break stateLoop;
							}
						}
					}
				}
				if (bks.size() == c) {
					System.out.println("INFINITE LOOP, DUMPING GOOD:");
					for (Brick b: reverseOrderedBricks) {
						System.out.println(b);
					}
					System.out.println("INFINITE LOOP, DUMPING BAD:");
					for (Brick b: bks) {
						System.out.println(b);
					}
					System.exit(0);
				}
			}
			bks.clear();
			rt = crt;
			Collections.reverse(reverseOrderedBricks);
			bks.addAll(reverseOrderedBricks);
			for (Brick b : bks) {
				if (b.rt != rt) {
					if (actions.size() != 0) {
						int rotate = b.rt - rt;
						System.out.println("CURR ROTATION : "+ rt + ", BRICK : " + b.rt + " MOVE: " + rotate);
						if (rotate < 0) {
							for (int i = 0; i > rotate; i--) {
								actions.add(new Rotate(false));
							}
						} else {
							for (int i = 0; i < rotate; i++) {
								actions.add(new Rotate(true));
							}
						}
					}
					rt = b.rt;
				}
				if (b.z != z) {
					actions.add(new GoToZ(b.z));
					z = b.z;
				}
				actions.add(new PlaceBrick(b.type, b.getYP(), b.getXP()));
				System.out.println(b);
			}
		}

		// final rotate to show object
		actions.add(new Rotate(false));

		System.out.println("Printer Instructions");
		int c = 0;
		for (Action a : actions) {
			System.out.println("" + c + ": " + a);
			c++;
		}
		
		// now that we have all the bricks, generate Actions
		return actions;
	}

	public static List<List<Brick>> getZLists(List<Brick> all, int maxz) {
		List<List<Brick>> bricks = new ArrayList<List<Brick>>();
		for (int i = 0; i <= maxz; i++) {
			bricks.add(new ArrayList<Brick>());
		}
		for (Brick b : all) {
			bricks.get(b.z).add(b);
		}
		return bricks;
	}

}
